#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Пример работы с iKKM USB Serial API
Документация на ikkm.kz/apidoc

Для реальной интеграции, не рекомендуем использовать Python.
Данный скрипт приведен лишь для примера, по причине совместимости
с Windows, OSX, Linux, BSD.

"""

import serial
import serial.tools.list_ports as port_list # for device find
from inscriptis import get_text as html_to_text # human readable html to text output
import json
from configparser import ConfigParser
import requests
import logging

# logger setup
logger = logging.getLogger('ikkm_usb_sample')
logger.addHandler(logging.StreamHandler())
logging.basicConfig(level=logging.DEBUG, format='[%(asctime)s %(levelname)s] %(message)s', datefmt='%H:%M:%S')

# конфиг для хранения настроек подключения
config = ConfigParser()
config.read('config.ini')

# для сохранения API key
def saveApi(a):
    config.set("main", "apikey", a)
    with open("config.ini", "w") as f:
        config.write(f)

# запрос для iKKM


def main():
    # Подключение к iKKM
    # VendorId/ProductId = 067b:2303
    search_port_name = list(port_list.grep("067b:2303"))
    if not search_port_name:
      logging.info("Error: iKKM not found!")
      quit()
    ikkm_port = search_port_name[0][0] # Берем первый попавшийся iKKM
    ser = serial.Serial(ikkm_port, 115200) # Соединяемся с iKKM

    # См файл requests.py
    # ************************************************************
    req = requests.getZreportRequest(config.get("main", "apikey"))
    # ************************************************************
    # Отправляем запрос
    request_ready = json.dumps(req)+chr(0x0a) # делаем json в string и добавляем 0x0a в конец
    logging.info("Sending data...")
    ser.write(str.encode(request_ready)) # отправляем данные в iKKM

    # Чтение ответа
    logging.info("Data sent, Waiting for a reply...")
    ser.reset_input_buffer()
    ser.reset_output_buffer()
    while True:
        reading = ser.readline()
        reading = reading.decode('utf-8')
        logging.info(reading)
        answer = json.loads(reading)
        # проверяем ответ
        if "result" not in answer:
            logging.critical("Error: no reply from iKKM")
            quit() # не ждем Push сообщения, выходим
        elif (answer["result"]) == "400":
            logging.critical("Error: %s %s",answer["result"],answer["resultText"])
            quit() # не ждем Push сообщения, выходим
        elif (answer["result"]) == "203":
            logging.warning("Warning,repeated request: %s %s",answer["result"],answer["resultText"])
            if "body" in answer: saveApi(answer["body"]) # обновляем apikey на новый
            quit() # не ждем Push сообщения, выходим
        else:
            # сохраняем новый apikey
            if "body" in answer:
                if len(answer["body"]) == 8:
                    saveApi(answer["body"])
            logging.info("Got Success Result, %s", answer["result"])
            break;

    logging.info("Waiting for a Push messages:")
    # Ожидание Push сообщения
    while True:
        reading = ser.readline()
        reading = reading.decode('utf-8')
        #logging.info(reading)
        answer = json.loads(reading)
        if answer["replyMethod"][:1] == "@":
            logging.info("Got PUSH message: %s", answer["replyMethod"])
        if "print" in answer:
            # Печать виртуального принтера (HTML -> Text)
            print(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>")
            print(html_to_text(answer["print"]))
            print("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<")
            continue
        # Fiscal special
        if "cheqId" in answer:
            logging.info("Cheque ID: %s", answer["cheqId"])
        if "ofdCode" in answer:
            logging.info("OFD Ticket code is: %s",answer["ofdCode"])
        # BANK special
        if "transactionResult" in answer:
            logging.info("Bank payment result: %s",answer["transactionResult"])
        if "chequeId" in answer:
            logging.info("Bank chequeId: %s", answer["chequeId"])
        # Finish everything
        if answer["replyMethod"][-4:] == "done":
            logging.info("*** Received finalle: %s ***",answer["replyMethod"])
            break


if __name__ == "__main__":
    main()